/**
 * Provided with a config file pointing to package resources, bundles a daemon which can be packaged in an application.
 */

// An enclosing app would include a call to ./bin/build-mac
// Use it to build that app once it is built:
// ~/wt-builds/node/node-v16.3.0-darwin-x64/bin/node bin/bundler.js macos/process-scanning-lib/package/config.json

// Rebuild command:
// ./bin/build-mac && ~/wt-builds/node/node-v16.3.0-darwin-x64/bin/node bin/bundler.js macos/process-scanning-lib/package/config.json

"use strict";

function copyDirectorySync(source, target) {
  const fs = require("fs");
  const path = require("path");
  if (!fs.existsSync(target)) {
      fs.mkdirSync(target);
  }

  const files = fs.readdirSync(source);
  for (const file of files) {
      const sourcePath = path.join(source, file);
      const targetPath = path.join(target, file);

      if (fs.lstatSync(sourcePath).isDirectory()) {
          copyDirectorySync(sourcePath, targetPath);
      } else {
          fs.copyFileSync(sourcePath, targetPath);
      }
  }
}

async function doWork() {
  const fs = require("fs");
  const path = require("path");
  const execSync = require("child_process").execSync;

  const cwd = process.cwd();
  const configFilepath = path.resolve(cwd, process.argv[2]);
  console.log("Using config file: ", configFilepath);

  let configRaw = fs.readFileSync(configFilepath);
  let config = JSON.parse(configRaw);
  console.log("Using config:", JSON.stringify(config, null, 2));

  const daemonBundleId = config["daemonBundleId"];
  const daemonExecutableName = config["daemonExecutableName"];

  const daemonInfoPlistPath = path.resolve(cwd, config["daemonInfoPlistPath"]);
  const daemonEntitlementsPath = path.resolve(
    cwd,
    config["daemonEntitlementsPath"]
  );
  const daemonProvisionProfilePath = path.resolve(
    cwd,
    config["daemonProvisionProfilePath"]
  );
  const daemonLaunchdPlistPath = path.resolve(
    cwd,
    config["daemonLaunchdPlistPath"]
  );
  const targetDir = path.resolve(cwd, config["target"]);
  const daemonBundlePath = path.resolve(targetDir, `${daemonBundleId}.app`);
  const daemonContentsPath = path.resolve(daemonBundlePath, "Contents");
  const daemonMacOSPath = path.resolve(daemonContentsPath, "MacOS");
  const daemonExecutablePath = path.resolve(
    daemonMacOSPath,
    daemonExecutableName
  );
  const daemonLibraryPath = path.resolve(daemonContentsPath, "Library");
  const bundledBdRedistPath = path.resolve(daemonLibraryPath, "bd-redist");
  const bundledBdRedistx86_64Path = path.resolve(bundledBdRedistPath, "mac-10.7-x86_64");
  const bundledDaemonLaunchdPlistPath = path.resolve(
    daemonLibraryPath,
    "launchd.plist"
  );
  const bundledDaemonInstallerPath = path.resolve(
    daemonLibraryPath,
    "installer"
  );
  const bundledDaemonUninstallerPath = path.resolve(
    daemonLibraryPath,
    "uninstaller"
  );
  const bundledVersionFilePath = path.resolve(daemonLibraryPath, "version");

  console.log("Removing target dir:", targetDir);
  fs.rmSync(targetDir, { recursive: true, force: true });

  console.log("Creating target dir structure");
  fs.mkdirSync(daemonMacOSPath, { recursive: true });
  fs.mkdirSync(daemonLibraryPath, { recursive: true });
  fs.mkdirSync(bundledBdRedistPath, { recursive: true });

  console.log("Copying executables");
  fs.copyFileSync(
    `${__dirname}/../native/executables/process-scanning-daemon`,
    daemonExecutablePath
  );

  fs.copyFileSync(
    `${__dirname}/../native/executables/daemon.installer`,
    bundledDaemonInstallerPath
  );

  fs.copyFileSync(
    `${__dirname}/../native/executables/daemon.uninstaller`,
    bundledDaemonUninstallerPath
  );

  console.log("Copying bd-redist");
  copyDirectorySync(
    `${__dirname}/../native/process-scanning-daemon-libs/bd-redist/mac-10.7-x86_64`,
    bundledBdRedistx86_64Path
  );

  console.log("Filling in bd-redist/license.txt")
  fs.writeFileSync(path.resolve(bundledBdRedistPath, "license.txt"), `${process.env["WTQW_MACOS_BD_SDK_LICENSE"]}\r\n`)

  console.log("Copying Info.plist files");
  fs.copyFileSync(
    daemonInfoPlistPath,
    path.resolve(daemonContentsPath, "Info.plist")
  );

  console.log("Copying embedded.provisionprofile files");
  fs.copyFileSync(
    daemonProvisionProfilePath,
    path.resolve(daemonContentsPath, "embedded.provisionprofile")
  );

  console.log("Copying daemon launchd plist");
  fs.copyFileSync(daemonLaunchdPlistPath, bundledDaemonLaunchdPlistPath);

  console.log("Writing version file");
  fs.writeFileSync(bundledVersionFilePath, process.env.WTQW_BUILD_VERSION);

  const notarize = require("@electron/notarize").notarize;
  const appleId = process.env["WTQW_MACOS_NOTARIZATION_USERNAME"];
  const appleIdPassword = process.env["WTQW_MACOS_NOTARIZATION_PASSWORD"];
  const ascProvider = process.env["WTQW_MACOS_TEAM_ID"];
  const certName = process.env["WTQW_MACOS_CERTIFICATE_NAME"];

  console.log("Signing daemon");
  execSync(
    `
      codesign --sign "${certName}" \\
      --force --timestamp --options runtime --verbose \\
      --entitlements "${daemonEntitlementsPath}" \\
      "${path.resolve(daemonContentsPath, "embedded.provisionprofile")}"
  `,
    { stdio: "inherit" }
  );
  execSync(
    `
      codesign --sign "${certName}" \\
      --force --timestamp --options runtime --verbose \\
      --entitlements "${daemonEntitlementsPath}" \\
      "${daemonBundlePath}"
  `,
    { stdio: "inherit" }
  );
  execSync(
    `
      codesign --verify --deep --strict --verbose=2 "${daemonBundlePath}"
  `,
    { stdio: "inherit" }
  );
  execSync(
    `
      codesign -d --entitlements :- "${daemonBundlePath}"
  `,
    { stdio: "inherit" }
  );

  if (process.env["NOTARIZE"]) {
    console.log("Notarizing daemon");
    await notarize({
      appPath: daemonBundlePath,
      tool: "notarytool",
      appleId,
      appleIdPassword,
      teamId: ascProvider,
    });
  }
}

doWork().then(() => console.log("Process Scanning Daemon Bundler is done!"));
